<?php

defined( 'ABSPATH' ) || exit();

class Radio_Player_Statistics {
	private static $instance = null;

	public function __construct() {
		add_action( 'wp_ajax_radio_player_get_statistics', array( $this, 'get_statistics' ) );

		add_action( 'wp_ajax_radio_player_insert_log', array( $this, 'insert_log' ) );
		add_action( 'wp_ajax_nopriv_radio_player_insert_log', array( $this, 'insert_log' ) );

		add_action( 'wp_dashboard_setup', [ $this, 'dashboard_widgets' ] );
	}

	public function dashboard_widgets() {

		wp_add_dashboard_widget( 'radio_player_chart', esc_html__( 'Radio Player Stats', 'radio-player' ), [
			$this,
			'render_dashboard_widget'
		] );

		// Globalize the metaboxes array, this holds all the widgets for wp-admin.
		global $wp_meta_boxes;

		// Get the regular dashboard widgets array
		// (which already has our new widget but appended at the end).
		$default_dashboard = $wp_meta_boxes['dashboard']['normal']['core'];

		// Backup and delete our new dashboard widget from the end of the array.
		$wp_dark_mode_widget_backup = array( 'radio_player_chart' => $default_dashboard['radio_player_chart'] );
		unset( $default_dashboard['radio_player_chart'] );

		// Merge the two arrays together so our widget is at the beginning.
		$sorted_dashboard = array_merge( $wp_dark_mode_widget_backup, $default_dashboard );

		// Save the sorted array back into the original metaboxes.
		$wp_meta_boxes['dashboard']['normal']['core'] = $sorted_dashboard;
	}

	public function render_dashboard_widget() { ?>
        <div id="radio-player-chart-widget"></div>
	<?php }

	public function insert_log() {
		$player_id = ! empty( $_REQUEST['id'] ) ? intval( $_REQUEST['id'] ) : '';

		if ( ! $player_id ) {
			wp_send_json_error( 'Empty player ID' );
		}

		$user_ip = radio_player_get_user_ip();
		$date    = date( 'Y-m-d' );

		$unique_id = md5( $date . $player_id . $user_ip );

		global $wpdb;
		$table = $wpdb->prefix . 'radio_player_statistics';

		$sql = "INSERT INTO 
                        {$table} (`player_id`,`unique_id`, `user_ip`, `created_at`, `updated_at`) 
                    VALUES 
                        (%d, %s, %s, %s, %s)
                    ON DUPLICATE KEY UPDATE
                        `count` = `count` + 1,
                        `updated_at` = VALUES (updated_at)  
                ";

		$wpdb->query( $wpdb->prepare( $sql, [
			$player_id,
			$unique_id,
			$user_ip,
			$date,
			$date
		] ) );


		wp_send_json_success();
	}

	public function get_logs( $start_date, $end_date ) {
		global $wpdb;
		$table = $wpdb->prefix . 'radio_player_statistics';

		// Prepare the SQL statement to prevent SQL injection
		$sql = $wpdb->prepare(
			"SELECT player_id, count, user_ip, DATE_FORMAT(`created_at`, '%%Y-%%m-%%d') AS `date`
        FROM {$table}
        WHERE `created_at` BETWEEN %s AND %s
        ORDER BY `date`
        LIMIT 0, 99999",
			$start_date, $end_date
		);

		// Execute the prepared statement and return the results
		return $wpdb->get_results( $sql );
	}

	public function get_statistics() {
		if ( ! check_ajax_referer( 'radio-player', 'nonce', false ) ) {
			wp_send_json_error( 'Invalid nonce', 403 );

			return;
		}

		$start_date = ! empty( $_POST['start_date'] ) ? sanitize_text_field( $_POST['start_date'] ) : date( 'Y-m-d', strtotime( '-1 month' ) );
		$end_date   = ! empty( $_POST['end_date'] ) ? sanitize_text_field( $_POST['end_date'] ) : date( 'Y-m-d' );

		// Validate date format to ensure they are correct
		$date_format      = 'Y-m-d';
		$start_date_valid = DateTime::createFromFormat( $date_format, $start_date ) !== false;
		$end_date_valid   = DateTime::createFromFormat( $date_format, $end_date ) !== false;

		if ( ! $start_date_valid || ! $end_date_valid ) {
			wp_send_json_error( 'Invalid date format. Please use YYYY-MM-DD.', 400 );

			return;
		}

		// Adjust end_date to include the entire day
		$end_date .= ' 23:59:59';

		$logs = $this->get_logs( $start_date, $end_date );

		$data = [
			'logs'        => $logs,
			'top_players' => $this->get_top_players( $start_date, $end_date ),
		];

		wp_send_json_success( $data );
	}

	public function get_top_players( $start_date, $end_date ) {
		global $wpdb;

		$table = $wpdb->prefix . 'radio_player_statistics';

		$page     = 1;
		$per_page = 50;
		$offset   = $per_page * ( $page - 1 );

		// Prepare SQL with placeholders for variables
		$sql = $wpdb->prepare(
			"SELECT DISTINCT `player_id`,
            COUNT(DISTINCT `user_ip`) AS `total_uniques`,
            SUM(`count`) AS `total_sessions`
        FROM {$table}
        WHERE `updated_at` BETWEEN %s AND %s
        GROUP BY `player_id`
        ORDER BY `total_sessions` DESC, `total_uniques` DESC
        LIMIT %d, %d",
			$start_date, $end_date, $offset, $per_page
		);

		$logs        = $wpdb->get_results( $sql );
		$top_players = [];

		if ( ! empty( $logs ) ) {
			foreach ( $logs as $log ) {
				$player_id = $log->player_id;

				// This function call suggests an external utility function. Ensure it's optimized for repeated calls.
				$player = radio_player_get_players( $player_id );

				// Checking for the title before deletion seems arbitrary. Ensure this is the desired behavior.
				if ( empty( $player['title'] ) ) {
					$wpdb->delete( $table, [ 'id' => $player_id ] );
					continue;
				}

				$log->title    = $player['title'];
				$top_players[] = $log;
			}
		}

		return $top_players;
	}

	public static function view() { ?>
        <div id="radio-player-statistics"></div>
	<?php }

	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}
}

Radio_Player_Statistics::instance();
